/**
* \file: Statistics.cpp
*
* \version: $Id:$
*
* \release: $Name:$
*
* <brief description>.
* <detailed description>
* \component: CarPlay
*
* \author: J. Harder / ADIT/SW1 / jharder@de.adit-jv.com
*
* \copyright (c) 2013-2014 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/

#include "Statistics.h"
#include "Common.h"
#include "control/PfCfgConfiguration.h"

using namespace std;

namespace adit { namespace carplay
{

// external access for AirPlay code
extern "C" void SetCarPlayDeviceID(char* inAddress)
{
    Statistics::Instance().SetDeviceID(inAddress);
}

Statistics Statistics::singleton; // initialized by constructor

static const char* yesNo(bool inValue)
{
    return inValue ? "yes" : "no";
}

Statistics::Statistics()
{
    lock = PTHREAD_MUTEX_INITIALIZER;
    deviceID = "unknown";
    config = nullptr;
    serverCount = 0;
    sessionCount = 0;
}

void Statistics::SetDeviceID(const string& inDeviceID)
{
    pthread_mutex_lock(&lock);
    deviceID = inDeviceID;
    pthread_mutex_unlock(&lock);
}

void Statistics::AddServer(const string& inConfigFile, IStatisticsProvider& inConfig)
{
    pthread_mutex_lock(&lock);
    serverCount++;
    configFile = inConfigFile;
    config = &inConfig;
    pthread_mutex_unlock(&lock);
}

void Statistics::RemoveServer()
{
    pthread_mutex_lock(&lock);
    serverCount--;
    pthread_mutex_unlock(&lock);
}

void Statistics::AddSession(void* inSession)
{
    (void)inSession;

    pthread_mutex_lock(&lock);
    sessionCount++;
    pthread_mutex_unlock(&lock);
}

void Statistics::RemoveSession(void* inSession)
{
    (void)inSession;

    pthread_mutex_lock(&lock);
    sessionCount--;
    pthread_mutex_unlock(&lock);
}

void Statistics::SetLatestModeChange(void* inSession, const string& inModeChange)
{
    (void)inSession;
    auto stats = &sessionStatistics;

    pthread_mutex_lock(&lock);
    stats->modeChange = inModeChange;
    pthread_mutex_unlock(&lock);
}

void Statistics::SetAVChannel(void* inSession, AVChannel inChannel, bool inActive)
{
    (void)inSession;
    auto stats = &sessionStatistics;

    pthread_mutex_lock(&lock);
    stats->active[inChannel] = inActive;
    pthread_mutex_unlock(&lock);
}

void Statistics::DumpStatistics()
{
    pthread_mutex_lock(&lock);

    dumpToFile(stdout);
    FILE* file = fopen("/tmp/carplay-stats.txt", "a+");
    if (file != nullptr)
    {
        dumpToFile(file);
        fclose(file);
    }

    pthread_mutex_unlock(&lock);
}

void Statistics::dumpToFile(FILE* inFile)
{
    fprintf(inFile, "* CarPlay info *\n");

    // print versions
    fprintf(inFile, "CarPlay git tag                     : %s\n", CarPlaySourceVersion.c_str());
    fprintf(inFile, "Communications plug-in core version : %s\n",
            CarPlayCommunicationsPluginVersion.c_str());
    fprintf(inFile, "developed with iOS                  : %s\n",
            CarPlayDevelopedWithiOSVersion.c_str());

    // device ID
    fprintf(inFile, "device ID                           : %s\n", deviceID.c_str());
    fflush(inFile);

    // server
    fprintf(inFile, "active servers                      : %d\n", serverCount);
    if (serverCount > 0)
    {
        fprintf(inFile, "config file                         : %s\n", configFile.c_str());
        if (config != nullptr)
        {
            // configuration
            auto lines = config->GetStatistics();
            for (auto line : lines)
            {
                fprintf(inFile, "config: %s\n", line.c_str());
            }
        }
    }

    // sessions
    fprintf(inFile, "active sessions                     : %d\n", sessionCount);

    auto stats = (sessionCount > 0) ? &sessionStatistics : nullptr;
    if (stats != nullptr)
    {
        fprintf(inFile, "latest mode change                  : %s\n", stats->modeChange.c_str());
        fprintf(inFile, "video channel                       : %s\n",
                yesNo(stats->active[AVChannel_Video]));
        fprintf(inFile, "main audio channel                  : %s\n",
                yesNo(stats->active[AVChannel_MainAudio]));
        fprintf(inFile, "alternate audio channel             : %s\n",
                yesNo(stats->active[AVChannel_AltAudio]));
    }

    fprintf(inFile, "* end of CarPlay info *\n");
    fflush(inFile);
}

} } // namespace adit { namespace carplay
